// Load and process YAML files
/* global sc, Packages */

function moduleFunction(config) {
  // Initialize $$ with provided config or default
  let $$ = config || require('./utils/debug')({ debug: 'cache', path: module.id });
  $$.fs = require('./utils/fs')($$);
  let jsyaml = require('cdev/js-yaml');

  /**
   * Reads a YAML file and parses its content.
   *
   * @param {string} yamlFile - The path to the YAML file.
   * @returns {Object} - The parsed YAML content.
   */
  function read(yamlFile) {
    let content = $$.fs.readFile(yamlFile);
    return jsyaml.load(content);
  }

  /**
   * Writes a JavaScript object to a YAML file.
   *
   * @param {Object} yamlObject - The object to serialize.
   * @param {string} yamlFile - The path to the YAML file.
   */
  function write(yamlObject, yamlFile) {
    let yamlContent = dump(yamlObject);
    $$.fs.writeFile(yamlFile, yamlContent);
  }

  /**
   * Parses a YAML string.
   *
   * @param {string} yamlContent - The YAML content as a string.
   * @returns {Object} - The parsed YAML content.
   */
  function load(yamlContent) {
    return jsyaml.load(yamlContent);
  }

  /**
   * Serializes a JavaScript object to a YAML string.
   *
   * @param {Object} yamlObject - The object to serialize.
   * @returns {string} - The YAML string.
   */
  function dump(yamlObject) {
    return jsyaml.dump(yamlObject);
  }

  // Return the module functions
  return {
    read: read,
    write: write,
    load: load,
    dump: dump
  };
}

// Initialize $$ when the module is required without parameters
let moduleExports = moduleFunction();

// Export the module function and its methods
module.exports = Object.assign(moduleFunction, moduleExports);
